//       file: hushhush.asm (also creates the "shooshoo" and the "5" segments)
//    project: Turrican2 plain -TDE v1.0
// asm format: Kick Assembler
// ---------------------------------------------------------------------------
// replacement loader for original Turrican2 disk version "##.prg" 
// goal is to create a plain version from original Turrican2 disks without drive-code
// and without hidden encrypted files and without fast-loader code
// to be able to run it in an emulator without true drive emulation
// because it plays great with a gamepad on android, having space on gamepad
// button to quickly access the energy lines and wheel 
// but switching disk images sucks in retroarch

// February 2019

* = $0326 // CHROUT vector address

.word prgstart                                  // autostart via kernal CHROUT vector hook, aka "READY" trap
.word $f6ed, $f13e, $f32f, $fe66, $f4a5, $f5ed  // other kernal vectors
memcfg: .byte $00 // for storing/restoring memory configuration

// information about files from original disks:
// final address indicates that the game relocates the loaded data to elsewhere in RAM
// 
/*                | original filename |           | temporary or  |
    new filename  | or loading call   | load addr | final address | notes / after load
    --------------+-------------------+-----------+---------------+-------------------
    0               hidden              0800 6fff
    
    1               hidden              7000 ab17   an interleaved
                                                    part of it ends
                                                    up deinterleaved
                                                    at cc00 cfff
     files 0 and 1 contain the entire intro graphics, music and scrolltext
     data and the code to run it as well
    2               hidden              0400 67ff                   load call at .C:cc7c
      2 appears to always stay in RAM and is never overwritten by any
      other loads
    3               hidden              6c00 b932                   load call at .C:ccb4
    
    4               hidden              67ff 68ff                   load call at .:ccbd
                                                                    to set address and
                                                                    at .C:ccc4 for load                                                                    
      4 contains the highscore table
      loaded by another, third loading mechanism that TurricanI did not have
      which allows setting a manual target address prior to sending the
      load request to the drive-code (appears to be only used for loading
      the highscore table)
    ----
    5 is not part of original turrican2 but a patch segment for high
      score saving routine because there was no room to fit it into
      the shooshoo segment
    ----

    A               hidden              9000 bf7f   d000 ff7f
      gets relocated after loading, first load after mainscreen
    B               hidden              6c00 c7ff
      second load after mainscreen
    A,B,..U,V       hidden              varies      varies 
      level sprites and level data
    W               hidden              9000 bfbf   d000 ff7f
    X               hidden              6c00 cfff
      W and X are part of "the end"
    
    
*/


// Enables creation of breakpoints
//
.var brk_file = createFile("breakpoints.txt")

.macro break() 
{
  .eval brk_file.writeln("break " + toHexString(*))
}

.macro break_at(address)
{
  .eval brk_file.writeln("break " + toHexString(address))
}

.function address_of(symbol)
{
  .return "$" + toHexString(symbol,4)
}

.macro printpc(where)
{
  .print where + ": $" + toHexString(*,4)
}

.macro printpcexplizit(what, where)
{
  .print what + ": " + address_of(where)
}

.function getfname(number)
{
  .return fnames + number * 1
}

.function getfnameL(number)
{
  .return fnamesL + number * 1
}

.macro patchfname(target, number) // patch filename at target to filename by index
{
  .var at = getfname(number)
  lda at
  sta target
}

.macro patch(target, value)
{
  lda #value
  sta target
}

.macro patchJMP(target, jmpaddress)
{
  lda #JMP_ABS
  sta target
  lda #<jmpaddress
  sta target+1
  lda #>jmpaddress
  sta target+2
}

.macro patchJSR(target, jmpaddress)
{
  lda #JSR_ABS
  sta target
  lda #<jmpaddress
  sta target+1
  lda #>jmpaddress
  sta target+2
}

.macro patchaddress(target, address)
{
  lda #<address
  sta target
  lda #>address
  sta target+1
}

.macro patchNOP(target, count)
{
  ldx #<target
  ldy #>target
  lda #count
  jsr sr_nopper
}

.macro patchNOP0(target, count)
{
  ldx #<target
  ldy #>target
  lda #count
  jsr sr_nopper0
}

.macro load0(number)
{
  .var at = getfname(number)
  ldx #<at
  ldy #>at
  jsr sr_load0
}

.macro loadL(number)
{
  .var at = getfnameL(number)
  ldx #<at
  ldy #>at
  jsr sr_loadB
}

.macro getfileLnamXY(number)
{
  .var at = getfnameL(number)
  ldx #<at
  ldy #>at
}

// for use with fnames in early lookup table
.macro patchCALLXY0(target, findex0)
{
  .var at = getfname(findex0) 
  ldx #<at
  stx target+1
  ldy #>at
  sty target+3
}

// for fnames from lookup table in shooshoo fnamesL
.macro patchCALLXYL(target, findexL)
{
  .var at = getfnameL(findexL) 
  ldx #<at
  stx target+1
  ldy #>at
  sty target+3
}


sr_load0: // replaces drive-code based loader with standard kernal loads
          // (changing this routine, requires changing ##.asm as well
          //  because the mechanism relies on these being at same RAM locs)
printpc("sr_load0")
lda $01 
sr_load_sm_mcfg1:
sta memcfg // store ram/rom cfg
and #%11111000
ora #%00000110 // mem cfg to: RAM, IO, KERNAL
sta $01//*/

sm_fname_length:
lda #$01     //			SUBROUTINE_LOAD0 (load by filename as listed in directory)
jsr $FFBD    // KERNAL SET FILENAME, length A, Y high address, X low address
lda #$00
ldx #$08
ldy #$01
jsr $FFBA    // KERNAL SET LOGICAL FILE PARAMS, A file number, X device, Y secondary address
lda #$00
tax
tay
jsr $FFD5    // KERNAL LOAD/VERIFY RAM FROM DEVICE, A 0 = load, X,Y load address if secondary address == 0

sr_load_sm_mcfg2:
lda memcfg
sta $01 // restore pre load ram/rom cfg*/

rts    
sr_load_after:

.label fnameSHOOSHOO = *
.text "SHOOSHOO"

.label fnames = *
.text "01" 


.label sr_loadB0  = $6900 // location of original Turrican2 loader B code  
                          // will be replaced by "shooshoo" segment   
.label shooshoo_max = $6bf8 // shooshoo must not grow beyond this because
                            // this will get clobbered by later loadings
                   // $6bf9 used by loader for some flag (is set to 01)      

// shooshoo will load here because at the real target location it would
// clobber the intro graphics there, so it will relocate itself after
// the intro is done and gone from screen
.label shooshoo_loadto = $ab20 
  

sr_nopper0:       // for early use with patchNOP0 before shooshoo is
    stx $02       // loaded and ready at its final destination
    sty $03
    tay
    dey
    lda #NOP
    lnop0:
    sta ($02),y
    dey
    bpl lnop0
    rts


.segment patch_for_3 [outPrg="5"]
* = $C170 // appears to be unused during mainscreen
printpc("segment_5_patch_ahead")
    segment_5_patch_ahead:
    // perform additional patches for mainscreen code here
    // which did not fit into shooshoo
    
    // skip "insert side 1..." after highscore
    // save question was answered
    patchJMP($92c1 ,$92e6) 
    
    // insert highscore saver replacement routine
    patchaddress($92fa, segment_5_save_highscore)
    
    rts
    
    .var line_adjust = 10
    segment5_replacement_irq_handler:    
    // check if it's a rasterline interrupt
    lda $d019
    and #%10000001
    and $d01A
    beq segment5_regular_irq
            jsr s5_determine_next_irq_line
            
            txa
            clc
            adc #line_adjust
            !: cmp $d012
            bpl !- // wait for the real irq line

            // the t2 handlers save and restore A,X,Y in zeropage
            // so there is no need to do that here
            sm_S5_jsr_target:
            printpc("sm_S5_jsr_target")
            // don't insert anything in this line! :P
            jsr $DEAD // call modified t2 handler here

            
    segment5_regular_irq:
    jsr $FFEA // kernal increment realtime clock
    lda #$00
    sta $91 // clear STOP key flag
    jmp $EA7E // onward to near end of kernal handler
              // but skip all the cursor/screen/tape motor/keyboard stuff



    s5_patch_mem_cfg_a_retmethod_x:
    sta $80ed // in IRQ_HANDLER_MAIN_80E2 (at rasterline 231)
    sta $822e // in IRQ_HANDLER_MAIN_8223 (at rasterline 153)
    sta $8b49 // in IRQ_HANDLER_MAIN_8B3E (at rasterline 202)
    stx $8222 // in IRQ_HANDLER_MAIN_80E2 (at rasterline 231)
    stx $8296 // in IRQ_HANDLER_MAIN_8223 (at rasterline 153)
    stx $8b75 // in IRQ_HANDLER_MAIN_8B3E (at rasterline 202)
    rts
    
    s5_patch_addresses_xy:
    stx $8175
    sty $8176
    stx $8234
    sty $8235
    stx $8b4f
    sty $8b50
    inx
    stx $817A
    sty $817B
    stx $8239
    sty $823A
    stx $8b54
    sty $8b55
    rts
    
    s5_determine_next_irq_line:
    lda sm_S5_jsr_target+2
    cmp #$8B
    bne !+
    ldx $8b57
    bne S5_next_irq_determined
    !:
    cmp #$82
    bne !+
    ldx $823c
    bne S5_next_irq_determined
    !:
    // cmp #$80 implicit if it was not one of the other two 
    ldx $817d
    S5_next_irq_determined:
    stx S5nextirqline
    rts
    
    .macro s5set_s_t(src,tgt)
    {
      ldx #<src
      ldy #>src
      jsr s5_set_src
      ldx #<tgt
      ldy #>tgt
      jsr s5_set_tgt
    }
    
    s5_set_src:
    stx s5s
    sty s5s+1
    rts
    
    s5_set_tgt:
    stx s5t
    sty s5t+1
    rts
    
    
    s5_s_t_downto0_y: // y must be set before call
    !:
    lda (s5s),y
    sta (s5t),y 
    dey
    bpl !-
    rts
    
    s5_s_t_0upto6d_y:
    ldy #$00
    !:
    lda (s5s),y
    sta (s5t),y
    iny
    cpy #$6e
    bne !-
    rts
    
    
printpc("segment_5_save_highscore (variables)")
    .var start_line = 48   // safe spot for stop of ints
    .var resume_line = 48 // safe spot for resume of ints
    .label s5s = $60
    .label s5t = s5s+2
    .label s5A = s5t+2
    .label s5X = s5A+1
    .label s5Y = s5X+1
    .label s5P = s5Y+1
    S5nextirqline: .byte $00
    S502a1: .byte $00  // for restoring 02a1 
    S5_90: .byte $00 // for kernal status save
    S5_kbscan_original: .byte $00 // for restoring t2 kbscan routine
    S5_irq_lines_original: .byte $00, $00, $00
    S5_6C06: .byte $00 // for restoring music related routine
    S5_6C03: .byte $00 // for restoring music related routine
printpc("segment_5_save_highscore")    
    segment_5_save_highscore:
    
    ldx #start_line
    !: cpx $d012
    bne !-
    lda $d011
    bmi !-
    
  
    // ------------------------------------------------------------------
    // halt t2 interrupt chain for a moment to patch it up
    // for replacements
    sei
    lda #$00  
    sta $D01A
    sta $D019
    lda $D019
    // interrupt chain halted
    

    // patch t2 interrupt chain for kernal friendly code flow
        // memcfg patches for RAM, IO, KERNAL
        lda #$36   
        // turn RTIs into RTSs
        ldx #RTS
        jsr s5_patch_mem_cfg_a_retmethod_x
        lda $94cc // first instruction in t2 kbscan routine
        sta S5_kbscan_original
        stx $94cc // disable with RTS during saving because it writes
                  // to zp kernal locations which could cause crashes
                  
        // still crashes, so disabling music routines as well
        lda $6C06
        sta S5_6C06
        stx $6C06
        lda $6C03
        sta S5_6C03
        stx $6C03
        
        // modify storage targets for next handlers in chain
        /*patchaddress($8175, sm_S5_jsr_target+1)
        patchaddress($817A, sm_S5_jsr_target+2) // in IRQ_HANDLER_MAIN_80E2
        patchaddress($8234, sm_S5_jsr_target+1)
        patchaddress($8239, sm_S5_jsr_target+2) // in IRQ_HANDLER_MAIN_8223
        patchaddress($8b4F, sm_S5_jsr_target+1)
        patchaddress($8b54, sm_S5_jsr_target+2) // in IRQ_HANDLER_MAIN_8B3E
        */
        // need to save bytes for more code here
        ldx #<(sm_S5_jsr_target+1)
        ldy #>(sm_S5_jsr_target+1)
        jsr s5_patch_addresses_xy
        
        // modify original irq_lines to a few lines earlier to adjust
        // for replacement irq handler overhead
        lda $817d
        sta S5_irq_lines_original
        sec
        sbc #line_adjust
        sta $817d
        lda $823c
        sta S5_irq_lines_original+1
        sec
        sbc #line_adjust
        sta $823c
        lda $8b57
        sta S5_irq_lines_original+2
        sec
        sbc #line_adjust
        sta $8b57
        
    
    // save t2 zp and vector area that will get clobbered by kernal save
            s5set_s_t($0090, zp_storage)
            jsr s5_s_t_0upto6d_y
            
            s5set_s_t($0314, vec_storage)
            ldy #$1f
            jsr s5_s_t_downto0_y
    
    
    // determine where to continue chain by examining currently scheduled
    // next handler in FFFF
            lda $FFFE
            sta sm_S5_jsr_target+1
            lda $FFFF
            sta sm_S5_jsr_target+2
            
            jsr s5_determine_next_irq_line
    
    // activate kernal, prepare kernal vectors, set user irq
            lda #$36
            sta $01   
            
            jsr $FF8A // call kernal routine to restore necessary vectors before save
                      // (clobbers 0314 0315 as well)
                      
            // now set user irq which will handle the modified t2 irq chain         
            ldx #<segment5_replacement_irq_handler         
            ldy #>segment5_replacement_irq_handler
            stx $0314
            sty $0315
            
            
  
    // resume modified t2 interrupt chain
            ldx S5nextirqline
            
            inc $d01a // resume raster interrupts 
            stx $d012 // at line of appropriate handler as determined above
            lda $d011
            and #$7f  // clear rcmp high bit
            sta $d011 
            cli       // resume interrupts
    // ------------------------------------------------------------------
    
    
    

    // perform highscore saving
    // ------------------------
            lda $02a1
            sta S502a1
               
            // clear some important zp vars which could cause save fail
            // if they contain turrican2 values
            lda #$00
            sta $9d // suppress kernal messages
            sta $94 // clear Flag: Serial Bus - Output Character buffered
                    // (flag caused re-loads to fail after continue-yes)
            sta $02a1 // clear "RS232 Enables"
            
            // better clear the entire kernal zp vars from any leftover
            // t2 stuff
            ldy #$6d
            !: sta $0090,y
            dey
            bpl !-
            
            lda #$01
            ldx #$08
            ldy #$01
            jsr $FFBA    // KERNAL SET LOGICAL FILE PARAMS, A file number, X device, Y secondary address 
            
            lda #$01
            getfileLnamXY(4) // "4" is the highscore file
            jsr $FFBD    // KERNAL SET FILENAME, length A, Y high address, X low address
            
            ldx #$ff
            ldy #$67
            stx s5s
            sty s5s+1
            iny
            lda #s5s
               
            jsr $FFD8    // KERNAL SAVE (A zp ptr to start, X,Y end address lo, hi)
          
            lda $90
            sta S5_90 // save kernal status for setting t2 expected
                      // flag before final rts
            
            lda S502a1
            sta $02a1




  
    ldx #resume_line
    !: cpx $d012
    bne !-
    lda $d011
    bmi !-

    // ------------------------------------------------------------------
    // halt t2 interrupt chain again to patch it back
    sei
    lda #$00  
    sta $D01A
    sta $D019
    lda $D019
    // interrupt chain halted
    
    // patch t2 interrupt chain back to kernal-free code flow
        // memcfg patches for RAM, IO, RAM
        lda #$35  
        // turn RTSs back into RTIs
        ldx #RTI
        jsr s5_patch_mem_cfg_a_retmethod_x
        ldx S5_kbscan_original
        stx $94cc // restore t2 kbscan routine
        
        // restore music related routines
        ldx S5_6C06
        stx $6C06
        ldx S5_6C03
        stx $6C03
        
        // modify storage targets for next handlers in chain
        /*patchaddress($8175, $FFFe)
        patchaddress($817A, $FFFF) // in IRQ_HANDLER_MAIN_80E2
        patchaddress($8234, $FFFe)
        patchaddress($8239, $FFFF) // in IRQ_HANDLER_MAIN_8223
        patchaddress($8b4F, $FFFe)
        patchaddress($8b54, $FFFF) // in IRQ_HANDLER_MAIN_8B3E
        */
        // need to save bytes for more code here
        ldx #$FE
        ldy #$FF
        jsr s5_patch_addresses_xy
    
        // restore original irq_lines
        lda S5_irq_lines_original
        sta $817d
        lda S5_irq_lines_original+1
        sta $823c
        lda S5_irq_lines_original+2
        sta $8b57
        
    // restore t2 zp and vector area that was clobbered by kernal save
            s5set_s_t(zp_storage, $0090)
            jsr s5_s_t_0upto6d_y
            
            s5set_s_t(vec_storage, $0314)
            ldy #$1f
            jsr s5_s_t_downto0_y
            
                // (these were 00 before)
                lda #$00
                ldy #$00
                !: sta s5s,y
                iny
                cpy #(s5P+1)
                bne !-
                
    // resume original t2 interrupt chain
    // restore last active handler from sm_S5_jsr_target+1,+2 into fffe ffff
    // and set rasterline depending on which handler it is, check hi byte
    // which is conveniently different for each handler
            lda sm_S5_jsr_target+1
            sta $FFFE
            lda sm_S5_jsr_target+2
            sta $FFFF
            
            jsr s5_determine_next_irq_line
    
    // de-activate kernal before resuming interrupts
            lda #$35
            sta $01 // to RAM, IO, RAM
            
            ldx S5nextirqline
            
            inc $d01a // resume raster interrupts 
            stx $d012 // at line of appropriate handler as determined above
                      // resume interrupts
            lda $d011
            and #$7f  // clear rcmp high bit
            sta $d011
            
            jsr $94cc // just update kbscan once before returning
                      // so that it does not hold what was saved
                      // there at call time
            
            cli       // resume interrupts
    // ------------------------------------------------------------------

    // don't forget to return the correct flag after all this work :D
    clc
    lda S5_90
    beq all_good
    sec // signals error to turrican2 code, so that it will display
        // an error message
    all_good:
    // phew! :D
    rts
    
.if (* > $c3ff) .print "--- ERROR: patch_for_3 (segment '5') is too big! ---"

.segment Default


.segment shooshoo [outPrg="shooshoo"]
printpcexplizit("sr_loadB (before more shooshoo routines)", sr_loadB)
* = shooshoo_loadto
shooshoo_reloc_self:
        // zp $43 to $46 appear to be unused by intro, so I use them as
        // pointers for relocation
        lda #<shooshoo_reloc_begin
        sta $43
        lda #>shooshoo_reloc_begin
        sta $44
        // ($43) source pointer
        
        lda #<sr_loadB0
        sta $45
        lda #>sr_loadB0
        sta $46
        // ($45) target pointer
        
        ldy #$00
        reloc_more:
        lda ($43),y
        sta ($45),y
        inc $43
        bne !+
        inc $44
        !:
        inc $45
        bne !+
        inc $46
        !:
        ldx $43
        cpx #<shooshoo_reloc_behind_end
        bne reloc_more
        ldx $44
        cpx #>shooshoo_reloc_behind_end
        bne reloc_more
        
        // although seemingly unused during intro..
        // ..still restoring $43 .. $46 to what was there before (all 00)
        lda #$00
        sta $43
        sta $44
        sta $45
        sta $46
        
        jmp shooshoo_patch_ahead_at_7c6c

shooshoo_reloc_begin:
.pseudopc sr_loadB0 { // pseudopc compiles this as if it was at sr_loadB 
// .pseudopc sr_loadB 
        sr_loadB: 
        lda $01 
        pha // sta shooshoo_sr_load_sm_mcfg2+1 // store ram/rom cfg
        and #%11111000
        ora #%00000110 // mem cfg to: RAM, IO, KERNAL
        sta $01//*/
        
        lda $02a1
        pha
        
        // clear some important zp vars which could cause load fail
        // if they contain turrican2 values
        lda #$00
        sta $9d // suppress kernal messages
        sta $94 // clear Flag: Serial Bus - Output Character buffered
                // (flag caused re-loads to fail after continue-yes)
        sta $02a1 // clear "RS232 Enables"
       
        shooshoo_sm_fname_length:
        lda #$01     //			SUBROUTINE_LOAD0 (load by filename as listed in directory)
        jsr $FFBD    // KERNAL SET FILENAME, length A, Y high address, X low address
        lda #$00
        ldx #$08
        ldy #$01
        jsr $FFBA    // KERNAL SET LOGICAL FILE PARAMS, A file number, X device, Y secondary address
        
        lda #$00
        tax
        tay
        jsr $FFD5    // KERNAL LOAD/VERIFY RAM FROM DEVICE, A 0 = load, X,Y load address if secondary address == 0
       
        pla
        sta $02a1
        
        
        pla
        sta $01 // restore pre load ram/rom cfg*/
        
        rts    
        shooshoo_sr_load_after:

        
    
    printpc("sr_nopper (in shooshoo)")
    sr_nopper:
    stx $02
    sty $03
    tay
    dey
    lda #NOP
    lnop:
    sta ($02),y
    dey
    bpl lnop
    rts
    
    // patches for IRQ handler used during loading screen
    printpc("(in shooshoo) fnamesL")
     fnamesL: .text "012345" 
                           // AB..WX levels and ending
    fnameCUR: .text "?"    // will be changed at runtime
    
    shooshoo_irq: // replacement interrupt handler to play nice with kernal
                // during first level loading
    printpc("shooshoo_irq")
    pha
    
    lda $01
    and #%00000111
    cmp #$04
    bne shooshoo_irq_ok // if IO available
    
    pla
    rti
    shooshoo_irq_ok:
            //inc $d020
            //inc $d021
            //dec $d020
            //dec $d021 // show some snow to signal activity
    
            lda $d019
            and #%10000001
            and $d01A
            beq shooshoo_after_raster_irq
    
    txa
    pha
    tya
    pha
    .var shooshoo_line_irq_first = 242 // during loading after intro
    .var shooshoo_line_irq_other = 106 // during other level and the end loading
            shooshoo_irq_sm_line:
            lda #shooshoo_line_irq_first 
            !: cmp $d012
            bpl !-
            
    .var shooshoo_loader_bg_sub_first = $cc08 // during first (after intro loading)
    .var shooshoo_loader_bg_sub_other = $61b1 // during other
            //sei
            shooshoo_irq_sm_sub:
            // don't insert anything in this line! :P
            jsr shooshoo_loader_bg_sub_first // draws the bg lines behind msg
            dec $d019 // ack
            //cli
            
    pla
    tay
    pla
    tax
            lda $01
            and #%00000111
            cmp #$06
            beq shooshoo_after_raster_irq // if kernal available
    pla
    shooshoo_irq_just_rti:
    rti
    shooshoo_after_raster_irq:
    pla
    
    jsr $FFEA // kernal increment realtime clock
    lda #$00
    sta $91 // clear STOP key flag
    jmp $EA7E // onward to near end of kernal handler
              // but skip all the cursor/screen/tape motor/keyboard stuff
    //jmp $EA31 // onward to kernal handler (will rti eventually)
 

    shooshoo_irq_tail: // needed for interrupt handler as subroutine
                     // during loadings between levels
    lda #$00
    sta $D020
    shooshoo_irq_just_rts:
    rts


                
   
    shooshoo_patch_loader:
    printpc("shooshoo_patch_loader")    
      sei
      lda #$36
      sta $01 // make sure kernal is on in case cli immediately fires an interrupt
     
      lda #%01111111
      sta $DC0D
      sta $DD0D
      lda $DC0D
      lda $DD0D 
      lda #%10000001
      sta $DC0D
      lda $DC0D // power-on interrupt IRQ state restored (need for kernel load)
      
      lda #$00  // disable raster compare interrupt
      sta $D019
      sta $D01A
      lda $D019 
      
      // set NMI handler
      /*lda #<shooshoo_irq_just_rti
      sta $fffa
      lda #>shooshoo_irq_just_rti
      sta $fffb*/ // don't !, causes problems with loading
      
      // set default kernal irq handler
      lda #$48
      sta $FFFE
      lda #$FF
      sta $FFFF
      
      // save turrican2 values before setting the irq vector
      lda $0314
      sta vec_storage
      lda $0315
      sta vec_storage+1
      
      // set user irq handler
      lda #<shooshoo_irq
      sta $0314
      lda #>shooshoo_irq
      sta $0315
            
      lda #$01
      sta $d019 // request raster irq..
      sta $d01A // request interrupt by rasterline only
      lda $d011
      and #$7f
      sta $d011 // clear raster cmp high
      shooshoo_loader_sm_line:
      lda #shooshoo_line_irq_first-3  // ..at this line (bit earlier than needed)
      sta $d012
      
    rts
    
    shooshoo_set_src: // set source pointer for copy loops
    stx pS
    sty pS+1
    rts
    
    shooshoo_set_tgt: // set target pointer for copy loops
    stx pT
    sty pT+1
    rts
    
    .macro set_s_t(src,tgt)
    {
      ldx #<src
      ldy #>src
      jsr shooshoo_set_src
      ldx #<tgt
      ldy #>tgt
      jsr shooshoo_set_tgt
    }
    
    shooshoo_s_t_downto0_y: // y must be set before call
    !:
    lda (pS),y
    sta (pT),y 
    dey
    bpl !-
    rts
    
    shooshoo_s_t_0upto6f_y:
    ldy #$00
    !:
    cpy #($D5-$90)
    bmi oky
    cpy #($D9-$90) // don't touch d5..d8 (or we'd clobber our own pointers)
    bpl oky
    bmi skipy
    oky:
    lda (pS),y
    sta (pT),y
    skipy:
    iny
    cpy #$70
    bne !-
    rts
    
    shooshoo_save_turrican2_values_from_zp_and_vecs_and_restore_kernal_vecs:
    sei
    set_s_t($0316, vec_storage+2)
    // only save 0316.. here because 0314 has already been altered
    // by shooshoo_patch_loader (and was saved before alteration in there)
    // +2 because 0314 and 0315 were saved in the first two locs
    ldy #$1d
    jsr shooshoo_s_t_downto0_y
    
    set_s_t($0090, zp_storage)
    jsr shooshoo_s_t_0upto6f_y
    
    lda $0314
    pha
    lda $0315
    pha
    
    jsr $FF8A // call kernal routine to restore necessary vectors before load
              // (clobbers 0314 0315 as well)
    pla
    sta $0315
    pla
    sta $0314
    
    rts
    
    shooshoo_restore_zp_and_vecs_to_turrican2_values:
    sei
    set_s_t(vec_storage, $0314)
    // unlike in the save function, here, all vectors down to 0314
    // are restored
    ldy #$1f
    jsr shooshoo_s_t_downto0_y
    
    
    set_s_t(zp_storage, $0090)
    jsr shooshoo_s_t_0upto6f_y
     
    rts


    shooshoo_prepare_level_load_irq:
    sei
        // use other line for interrupt
        patch(shooshoo_loader_sm_line+1, shooshoo_line_irq_other-3)
        sta $619b // same line for re-used original code part
        patch(shooshoo_irq_sm_line+1, shooshoo_line_irq_other)

        // use other subroutine for drawing the banner behind loading text
        patchaddress(shooshoo_irq_sm_sub+1, shooshoo_loader_bg_sub_other)
    
        jsr shooshoo_patch_loader
        
        // re-use original turrican2 code part for remaining settings 
        ldx #$00
        jmp $6192 // turns screen on, sprites off, requests irq by line
                  // acks vic irqs, cli
    // rts (is at end of jmp code)
    
    shooshoo_correct_reloc_pause_mem_cfg: // patched in at 54ff
    // to correct mem cfg for active irq handler
    //inc $01 // #$35 RAM, IO, RAM (what turrican2 did as it did not need kernal)
    //inc $01 // #$36 RAM, IO, KERNAL 4 bytes of code
    lda #$36
    sta $01  // also 4 bytes of code but more explicit
    cli
    rts
    
    shooshoo_ack_and_disable_vic_ints:
    // needed after loading and after loading first file of ending
    lsr $d019 // ack all vic ints (the dirty way)
    lda #$00
    sta $d01a // disable vic interrupts
    rts
    
    shooshoo_correct_end_file2_flow:
    ldx $03EF
    cpx #$0B // reached the end?
    bne !+
    jsr shooshoo_ack_and_disable_vic_ints
    !:
    rts
    
    
    shooshoo_load_2:
    // replaces load that happens after the intro finishes
    // (the one with the "LOADING" sprites coming up from the bottom)
    printpc("shooshoo_load2")
    jsr shooshoo_patch_loader
      
    loadL(2) // might cli during kernal calls
    
    // restore pla in original interrupt handler
    //lda #PLA
    //sta $cc72
    
    jmp shooshoo_patch_ahead_after_2
    
    shooshoo_reload_3:
    // replaces the reloading of the mainscreen after death or ending
    printpc("shooshoo_reload_3")

    sei // fix brk problem by overwriting int handler temporarily
    ldx #<shooshoo_irq_just_rti
    ldy #>shooshoo_irq_just_rti
    stx $fffe
    sty $ffff
    
    jsr shooshoo_patch_loader
    jsr shooshoo_ack_and_disable_vic_ints
    
    jsr shooshoo_save_turrican2_values_from_zp_and_vecs_and_restore_kernal_vecs
    
    loadL(3)
    loadL(5) // is a patch segment for highscore saving
             // for which there was not enough room in here
    jsr segment_5_patch_ahead

    jsr shooshoo_restore_zp_and_vecs_to_turrican2_values

    rts
    
    shooshoo_load_3:
    // replaces the load that happens after intro after music faded
    printpc("shooshoo_load3")
    // not patching loader here again, keeping the irq used there
    // as is
    
    loadL(3)
    
    rts
    
    shooshoo_load_4:
    // replaces the load that happens shortly after load_3
    printpc("shooshoo_load4")
    // still not patching loader here again, keeping the irq used there
    // as is
    
    loadL(4)
    
    rts // should end up at $7ff2 as in original turrican2 code


    
    
    shooshoo_load_fixed_fname:
    printpc("shooshoo_load_fixed_fname")    
    ldx #<fnameCUR
    ldy #>fnameCUR
    jsr sr_loadB
    rts
    
    
    shooshoo_setup_fixed_fname:
    printpc("shooshoo_setup_fixed_fname")
    lda $03ef // turrican2 level index
    tax // for evaluation by caller
    asl // * 2 because there are two files per level
    clc
    adc #('A')
    sta fnameCUR
    rts
    
    
    shooshoo_load_data0: // loads first file for level or end
    // for the cases where that file needs to be relocated
    printpc("shooshoo_load_data0")
    jsr shooshoo_save_turrican2_values_from_zp_and_vecs_and_restore_kernal_vecs
    jsr shooshoo_setup_fixed_fname
    jsr shooshoo_load_fixed_fname
    rts
    
    shooshoo_load_data0_1_2: // loads one or two files for level or end
    printpc("shooshoo_load_data0_1_2")
    // depending on whether data0 was already loaded or not
    // (determined by same lookup table which turrican2 uses for that)
    jsr shooshoo_setup_fixed_fname
    lda $6105,x // turrican2 lookup table has a 0 if level at index x
                // does not need to load anything which needs relocation
    bne already_loaded_data0
    
    // load first data file
    jsr shooshoo_save_turrican2_values_from_zp_and_vecs_and_restore_kernal_vecs
    jsr shooshoo_load_fixed_fname
    
    already_loaded_data0:
    // load second data file
    inc fnameCUR
    jsr shooshoo_load_fixed_fname
    jsr shooshoo_ack_and_disable_vic_ints
    jsr shooshoo_restore_zp_and_vecs_to_turrican2_values
    
    dec $01 // select RAM, IO, RAM now because turrican2 will
            // jmp into code loaded/moved under kernal later
    rts
   
    
    
    
    shooshoo_patch_ahead_at_7c6c: // entry, runs first after shooshoo
                                  // was relocated to its real destination
                                  // after intro played
    printpc("shooshoo_patch_ahead_at_7c6c")
    //
    // convert "loading" interrupt handler into subroutine
    patch($cc72, RTS)
    
    // patch rasterline for "loading" interrupt interrupt 
    // to a few lines earlier for overhead
    patch($7ca0, shooshoo_line_irq_first-3)
    
    // patch call to load "2"
    patchaddress($cc7d, shooshoo_load_2)
    
    // patch call to load "3"
    patchaddress($ccb5, shooshoo_load_3)
    
    // patch call to load "4" but also remove the turrican2 loader
    // extra steps involved in that
    patchNOP($ccb7, 13)
    patchaddress($ccc5, shooshoo_load_4) // highscore file
    
    // replicate original instructions at 7c6b before returning
    ldx #$07
    ldy #$67
    
    rts
    
// the previous parts of the segment 
// must not grow out of 6900-6bff or it will be clobbered    
// by subsequent loadings
.if (* > shooshoo_max) .print "-------- ERROR: shooshoo is in the way! --------"    
    
    // these patches below only need to run once, so they can be
    // clobbered after shooshoo ran them
    shooshoo_patch_ahead_after_2:
    .label pS = $d5 // (ind),y only works in zp // shooshoo_patch_ahead_after_2
    printpcexplizit("shooshoo pS", pS)
    .label pT = $d7 // (ind),y only works in zp //pS + 2
    // turrican2 uses d5..d8 for data relocations after loading, so
    // they can be used without disturbing anything game related
    printpcexplizit("shooshoo pT", pT)
    .label zp_storage = shooshoo_patch_ahead_after_2
    printpcexplizit("shooshoo zp_storage", zp_storage)
    .label vec_storage = zp_storage + $70
    printpcexplizit("shooshoo vec_storage", vec_storage)
    .label extra_storage = vec_storage + $20
    printpcexplizit("shooshoo extra_storage", extra_storage)
    
    printpc("shooshoo_patch_ahead_after_2 (will be clobbered after it ran)")
    // patch to skip "turn disk" code before loading next level or end
    patchJMP($5470, $54a2)
    
    // patch to replace original SUBROUTINE_PREPARE_LOADER_IRQ
    patchaddress($54b6, shooshoo_prepare_level_load_irq)
    // turn original irq handler during level loads into subroutine
    patch($61d1, RTS)
    
    // patch in a routine to set correct memory cfg for irq
    // while the relocation of loaded data pauses for the banner
    // to draw (need RAM, IO, KERNAL there for replacement irq)
    patchJSR($54ff, shooshoo_correct_reloc_pause_mem_cfg)
    // also correct the cfg after relocation is done
    patch($5510, $36) // need RAM, IO, KERNAL there
    
    // patch in vic ints disabling before loading second file of "the end"
    // (because the screen will turn off there as the contents would
    //  clobber the message on screen... and otherwise the blue banner
    //  would keep showing and jittering without a message)
    patchJSR($5514, shooshoo_correct_end_file2_flow)
    
    // patch to replace original data0 load for levels/end
    patchaddress($54ca, shooshoo_load_data0)
    
    // patch to replace original data12 load for levels/end
    patchaddress($5526, shooshoo_load_data0_1_2)
    
    // patch to skip "turn disk" code after game over or after end
    patchJMP($5410, $5451)
    
    // patch to replace call to reload "3" after game over or after end
    patchaddress($5452, shooshoo_reload_3)
    
    rts
     
printpc("after shooshoo(real)")
}
printpc("after shooshoo(before reloc)") 
shooshoo_reloc_behind_end:


.segment Default    

    
    // jsr shooshoo  // call the just loaded segment routine for more patches
    
    
.label prgstart = *
printpc("main")
main: 
    sei
// ---- section copied verbatim from original ##.prg -------------------------------------------------------
    cld
    lda #$CA
    sta $0326
    lda #$F1
    sta $0327 // original CHROUT vector restored
     
    lda #$00
    sta $D120 // BORDER COLOR BLACK	(VIC registers at $D000 repeat every 64 bytes until $D3FF)
    sta $D221 // BACKGROUND COLOR BLACK
    sta $D1D1 // SCREEN CONTROL REGISTER (same as $D011, because of repeated registers at 00,40,80,C0 (D1-C0==11)
              // v-scroll 0, screen height = 24 rows, screen off, extended bg mode off, raster interrupt line 0
    sta $D015 // disable sprites
    sta $9D   // suppress error/kernal messages
// ---- end of section copied verbatim from original ##.prg ------------------------------------------------    
    

    jmp show_readthis
fnameR:                 
.text "READTHIS"
    show_readthis:
    lda #$08
    sta sm_fname_length+1 // filename length to 8
    
    ldx #<fnameR
    ldy #>fnameR
    jsr sr_load0 // that file loads to screen 3 in bank 1
    
    sei
    lda #$01
    sta sm_fname_length+1 // filename length back to 1
    
    lda #$37
    sta $01 // basic, IO, kernal to active
    
    lda #DARK_GRAY
    sta $D020
    sta $D021
    lda #LIGHT_GRAY
    ldx #$00
    !:
    sta $d800,x
    sta $d900,x
    sta $da00,x
    sta $db00,x
    inx
    bne !- // set color ram
    
    // save current vic memory control reg on stack
    lda $d018
    pha
    
    // activate screen and charset 3 for displaying loaded READTHIS content
    lda #%00110111 // screen 3, charset 3
    sta $d018  
    lda #%00011011 // textmode, screen on, 25 rows, scroll-y 3
    sta $d011
    
    !:
    lda $dc01
    and #$10
    bne !- // wait for key-down: space key
    !:
    lda $dc01
    and #$10
    beq !- // wait for key-up: space key

    lda #$00
    sta $d011 // screen off

    // restore vic memory control reg
    pla
    sta $d011
     
    lda #$00
    sta $D120 // BORDER COLOR BLACK	(VIC registers at $D000 repeat every 64 bytes until $D3FF)
    sta $D221 // BACKGROUND COLOR BLACK
    sta $D1D1 // SCREEN CONTROL REGISTER (same as $D011, because of repeated registers at 00,40,80,C0 (D1-C0==11)
              // v-scroll 0, screen height = 24 rows, screen off, extended bg mode off, raster interrupt line 0
    
    load0(0)
    load0(1)

// ---- another section copied from ##.prg -----------------------------------------------------------------
    lda #$36
    sta $01					// select RAM,IO,KERNAL
    ldx #$00

    reloc_parts_of_1_loop:
    lda $A460,X			
    sta $CC00,X
    lda $A560,X
    sta $CD00,X
    lda $A660,X
    sta $CE00,X
    lda $A760,X
    sta $CF00,X
    inx
    bne reloc_parts_of_1_loop	
    // loop x[$00..$ff] relocates something from an interleaved layout 
    // to un-interleaved
		//      for X = 00 to 9f, data reads are from a*60+x  *[4..7]
    // but from X = a0 to ff, data reads are from a*00+x  *[4..7]
    // from A400..A85F (was in file "1") to CC00..CFFF (4 pages)
                    
    lda #$37
    sta $01					// select BASIC, IO, KERNAL
    jsr $FF84				// kernal routine: Init I/O Devices, Ports & Timers				
// ---- end of another section copied from ##.prg ----------------------------------------------------------    
    
    // load the extra segment, re-using RAM otherwise used by original
    // Turrican loaderB code
    lda #$08 // 8 letters in "shooshoo
    sta sm_fname_length+1
    ldx #<fnameSHOOSHOO
    ldy #>fnameSHOOSHOO
    jsr sr_load0 // load the extra segment below
    lda #$01     // back to 1 letter names
    sta sm_fname_length+1
    
    // patch ahead before jumping away into intro
    // ------------------------------------------
    
    // prevent shooshoo from being clobbered after intro
    patchNOP0($7c56, 18)
    
    // prevent irq vector from being set to $cc00 (shooshoo sets one later)
    patchNOP0($7c95, 10)
    // A: still holds opcode for NOP here, which is used to eliminate
    // the unneeded byte before the injection below
    sta $7c6b
    
    // inject first patch ahead routine at 7c6c after the
    // code targeted by the patches has been relocated to its final location
    // (by turrican2 code), shooshoo's first action will be to move itself
    // to $6900 where it can stay undisturbed by the rest of the game code
    patchJSR($7c6c, shooshoo_reloc_self)
             
    after_preloader_run_intro:
    jmp $7B3D // starts the intro which was in files dumped as  "0" and "1" 
              // (the entire intro! no loadings happen while it runs!)
    
.if (* > $0800) .print "-------- ERROR: hushhush is hushing too far! --------"
printpc("end of code")
 